<?php

require 'pojo/contact/Contact.php';
require 'pojo/contact/ContactGroup.php';
require 'pojo/contact/ExtraField.php';
require 'pojo/contact/ExtraFieldOption.php';
require 'pojo/contact/ExtraFieldValue.php';
require 'pojo/contact/ListPojo.php';
require 'pojo/contact/PaginationPojo.php';
require 'pojo/contact/SenderProfile.php';
require 'pojo/contact/post/ContactGroupPost.php';
require 'pojo/contact/post/ContactPost.php';
require 'pojo/contact/post/ContactRegister.php';
require 'pojo/contact/post/ExtraFieldValuePost.php';
require 'pojo/contact/ContactImportResponse.php';
require 'pojo/contact/ContactImportResponseList.php';
require 'pojo/error/ConstraintViolationError.php';

class EtherMailerApi {

    public static $instance;

    const METHOD_AUTO = 'METHOD_AUTO';
    const METHOD_CURL = 'METHOD_CURL';
    const METHOD_SOCKET = 'METHOD_SOCKET';

    public static function instance($apiKey, $method = self::METHOD_AUTO) {
        if (!isset(EtherMailerApi::$instance)) {
            // Create a new instance
            EtherMailerApi::$instance = new EtherMailerApi($apiKey, $method);
        }
        return EtherMailerApi::$instance;
    }

    /**
     * EtherMailerApi constructor.
     * @param string $apiKey
     * @param string $method
     */
    public function __construct($apiKey = '', $method = self::METHOD_AUTO) {
        if($method != self::METHOD_AUTO && $method != self::METHOD_CURL && $method != self::METHOD_SOCKET) {
            $this->throwException('Method '.$method.' is not supported!');
        }
        if($method == self::METHOD_AUTO) {
            if(function_exists('curl_init')) {
                $method = self::METHOD_CURL;
            } else {
                $method = self::METHOD_SOCKET;
            }
        }
        $this->apiKey = $apiKey;
        $this->method = $method;
    }

    protected $ethermailerApiHost = 'api.ethermailer.com';
    protected $apiKey = '';
    protected $method = '';

    const GET = 'GET';
    const POST = 'POST';
    const PUT = 'PUT';
    const DELETE = 'DELETE';

    /**
     * @return String
     */
    public function getUsername() {
        $response = $this->make_request(self::GET, '/user');
        
        if($response['status'] != 200) {
            return NULL;
        }
        return $response['data']['username'];
    }

    /**
     * @return PaginationPojo of type ContactGroup
     */
    public function getGroups() {
        $response = $this->make_request(self::GET, '/user/contact-groups/limit/50/offset/0');
        $paginationPojo = new PaginationPojo();
        if($response['status'] == 200) {
            $paginationPojo->totalItems = $response['data']['totalItems'];
            $items = array();
            if($response['data']['items'] != null) {
                foreach ($response['data']['items'] as $key => $value) {
                    $contactGroup = new ContactGroup();
                    $contactGroup->groupId = $value['groupId'];
                    $contactGroup->name = $value['name'];
                    $contactGroup->contactsCount = $value['contactsCount'];
                    $items[] = $contactGroup;
                }
                
            }
            $paginationPojo->items = $items;
            
        }
        return $paginationPojo;
    }

    /**
     * @return PaginationPojo of type ExtraField
     */
    public function getContactFields() {
        $response = $this->make_request(self::GET, '/user/extra-fields/limit/50/offset/0');
        $paginationPojo = new PaginationPojo();
        if($response['status'] == 200) {
            $paginationPojo->totalItems = $response['data']['totalItems'];
            $items = array();
            if($response['data']['items'] != null) {
                foreach ($response['data']['items'] as $key => $value) {
                    $extraField = new ExtraField();
                    $extraField->fieldId = $value['fieldId'];
                    $extraField->name = $value['name'];
                    $extraField->required = $value['required'];
                    $extraField->type = $value['type'];
                    if($value['options'] != null) {
                        $extraFieldOptions = array();
                        foreach ($value['options'] as $key2 => $value2) {
                            $extraFieldOption = new ExtraFieldOption();
                            $extraFieldOption->optionId = $value2['optionId'];
                            $extraFieldOption->text = $value2['text'];
                            $extraFieldOptions[] = $extraFieldOption;
                        }
                        $extraField->options = $extraFieldOptions;
                    }
                    $items[] = $extraField;
                }
            }
            $paginationPojo->items = $items;
        }
        return $paginationPojo;
    }

    /**
     * @return ListPojo of type SenderProfile
     */
    public function getSenderProfiles() {
        $response =  $this->make_request(self::GET, '/user/email-settings/sender-profiles');
        $listPojo = new ListPojo();
        if($response['status'] == 200) {
            $items = array();
            if ($response['data']['items'] != null) {
                foreach ($response['data']['items'] as $key => $value) {
                    $senderProfile = new SenderProfile();
                    $senderProfile->id = $value['id'];
                    $senderProfile->name = $value['name'];
                    $senderProfile->email = $value['email'];
                    $senderProfile->verified = $value['verified'];
                    $senderProfile->spf = $value['spf'];
                    $senderProfile->dkim = $value['dkim'];
                    $senderProfile->spfRecordName = $value['spfRecordName'];
                    $senderProfile->spfRecordValue = $value['spfRecordValue'];
                    $senderProfile->dkimRecordName = $value['dkimRecordName'];
                    $senderProfile->dkimRecordValue = $value['dkimRecordValue'];
                    $items[] = $senderProfile;
                }
            }
            $listPojo->items = $items;
        }
        return $listPojo;
    }

    /**
     * @param $contactRegister ContactRegister
     * @return mixed The response
     */
    public function registerContact($contactRegister) {
        return $this->make_request(self::POST, '/user/contact/register', NULL, json_encode($contactRegister));
    }

    /**
     * @param $contactPostList ContactPost[]
     * @return ContactImportResponseList
     */
    public function importContacts($contactPostList) {
        $response = $this->make_request(self::POST, '/user/contact/import', NULL, json_encode($contactPostList));
        $contactImportResponseList = new ContactImportResponseList();
        if($response['status'] == 200) {
            $contactImportResponseList = new ContactImportResponseList();
            $contactImportResponseList->totalCreated = $response['data']['totalCreated'];
            $contactImportResponseList->totalUpdated = $response['data']['totalUpdated'];
            $contactImportResponseList->totalErrors = $response['data']['totalErrors'];
            $contactImportResponseList->contactImportResponses = array();
            if($response['data']['contactImportResponses'] != null) {
                foreach($response['data']['contactImportResponses'] as $contactImportResponseTemp) {
                    $contactImportResponse = new ContactImportResponse();
                    $contactImportResponse->contactId = $contactImportResponseTemp['contactId'];
                    $contactImportResponse->contactEmail = $contactImportResponseTemp['contactEmail'];
                    $contactImportResponse->constraintViolationErrorList = array();
                    
                    if($contactImportResponseTemp['constraintViolationErrorList'] != null) {
                        foreach($contactImportResponseTemp['constraintViolationErrorList'] as $constraintViolationErrorTemp) {
                            $constraintViolationError = new ConstraintViolationError();
                            $constraintViolationError->objectName = $constraintViolationErrorTemp['objectName'];
                            $constraintViolationError->constraintName = $constraintViolationErrorTemp['constraintName'];
                            $constraintViolationError->description = $constraintViolationErrorTemp['description'];
                            $constraintViolationError->constraintProperties = array();
                            if(isset($constraintViolationErrorTemp['constraintProperties']) && $constraintViolationErrorTemp['constraintProperties'] != null) {
                                foreach($constraintViolationErrorTemp['constraintProperties'] as $propKey => $propValue) {
                                    $constraintViolationError->constraintProperties[$propKey] = $propValue;   
                                    unset($propKey);
                                    unset($propValue);
                                }
                            }
                            $contactImportResponse->constraintViolationErrorList[] = $constraintViolationError;
                            unset($constraintViolationErrorTemp);
                        }
                    }
                    
                    $contactImportResponseList->contactImportResponses[] = $contactImportResponse;
                    unset($contactImportResponse);
                    unset($contactImportResponseTemp);
                }
            }
        }
        return $contactImportResponseList;
    }

    protected function make_request($method, $path, $getParams = NULL, $postParams = NULL) {
        if($this->method == self::METHOD_CURL) {
            return $this->make_request_curl($method, $path, $getParams, $postParams);
        }
        if($this->method == self::METHOD_SOCKET) {
            return $this->make_request_socket($method, $path, $getParams, $postParams);
        }
        $this->throwException('No method to make request!');
    }

    protected function createFinalURL($path, $getParams) {
        $url = 'https://'.$this->ethermailerApiHost.$path;
        
        $getData = array();
        if ($getParams != null && is_array($getParams)) {
            $getData = $getParams;
        }
        unset($getParams);

        $getData = http_build_query($getData);

        $url = $url . '?' . $getData;

        return $url;
    }

    protected function make_request_curl($method, $path, $getParams = NULL, $postPayload = NULL) {
        $url = $this->createFinalURL($path, $getParams);

        //make resource
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_HEADER, true);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLINFO_HEADER_OUT, true);
        curl_setopt($ch, CURLOPT_USERAGENT, 'Ether Mailer API Client');
        curl_setopt($ch, CURLOPT_HTTPHEADER, array(
            'Authorization: Api '.$this->apiKey,
            'Accept: application/json',
            'Content-type: application/json'
        ));
        
        if ($method == 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            if($postPayload) {
                curl_setopt($ch, CURLOPT_POSTFIELDS, $postPayload);
            }
            
        }

        if ($method == 'PUT') {
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
            if($postPayload) {
                curl_setopt($ch, CURLOPT_POSTFIELDS, $postPayload);
            }
        }

        if ($method == 'DELETE') {
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'DELETE');
            if($postPayload) {
                curl_setopt($ch, CURLOPT_POSTFIELDS, $postPayload);
            }
        }

        //get contents
        $response = curl_exec($ch);
        $responseCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if($response == null) {
            return array(
                'status' => 'CANNOT_CONNECT',
                'data' => NULL
            );
        }

        list($response_headers, $response_body) = explode("\r\n\r\n", $response, 2);
        $array_headers = explode("\n", $response_headers);
        if (isset($array_headers[0])) {
            if ($array_headers[0] == 'HTTP/1.1 100 Continue') {
                list($response_headers, $response_body) = explode("\r\n\r\n", $response_body, 2);
            }
        }

        return array(
            'status' => $responseCode,
            'data' => json_decode($response_body, true)
        );
    }

    protected function make_request_socket($method, $path, $getParams = NULL, $postPayload = NULL) {
        $url = $this->createFinalURL($path, $getParams);

        $sh = @fsockopen($this->ethermailerApiHost, 843);
        if ($sh) {
            fputs($sh, "$method $url HTTP/1.0\r\n");
            fputs($sh, "User-Agent: Ether Mailer API Client\r\n");
            fputs($sh, "Authorization: Api ".$this->apiKey."\r\n");
            fputs($sh, 'Accept: application/json' ."\r\n");
            fputs($sh, 'Content-type: application/json' ."\r\n");

            if ($method == 'POST') {
                fputs($sh, 'Content-Length: ' . strlen($postPayload) ."\r\n");
                fputs($sh, "\r\n");
                fputs($sh, $postPayload);
            } else {
                fputs($sh, "\r\n");
            }

            $content = '';
            while (!feof($sh)) {
                $content .= fread($sh, 65536);
            }
            fclose($sh);

            list($headers, $content) = preg_split("/\R\R/", $content, 2);

            // remove 100 Continue header
            if (strpos($headers, '100 Continue') !== false) {
                list($headers, $content) = preg_split("/\R\R/", $content, 2);
            }

            $responseCode = substr($headers, 9, 3);
            return array(
                'status' => $responseCode,
                'data' => json_decode($content, true)
            );
        } else {
            return array(
                'status' => 'CANNOT_CONNECT',
                'data' => NULL
            );
        }
    }

    protected function throwException($message) {
        throw new Exception(get_class($this).': '.$message);
    }
}